
/*  @(#)x11.c 1.4 02/02/15
 *
 *  X11 dependent graphics routines used by popi.
 *  written by Rich Burridge - Sun Microsystems.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

#include <unistd.h>
#include "popi.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <poll.h>

#define POPI_BORDER_WIDTH  2

#define FRAME_MASK  (ExposureMask)

static unsigned short icon_image[] = {
#include "popi.icon"
};

#define POPI_INPUT  0       /* Types of input selected on. */
#define KEY_INPUT   1

static struct pollfd pfd[2];
static unsigned long npfd;         /* Number of file descriptors to monitor. */

static Atom protocol_atom, kill_atom;
static Colormap cmap = 0;
static Cursor busy_cursor, main_cursor;
static Display *dpy;
static GC gc, pix_gc;
static Pixmap memarea, popi_icon, load_icon();
static Window frame, root;
static XColor ccol;
static XEvent event;
static XGCValues gc_val;
static XSizeHints size;
static XWMHints wm_hints;

static unsigned long gc_mask;
static int screen;
static int xfd;                      /* Server connection file descriptor. */
static unsigned long backgnd, foregnd;
static unsigned long palette[CMAPLEN];
static int colsused = 0;

static int iscolor;                  /* Set if this is a color screen. */
static int ix = 0;                   /* Initial X position of the icon. */
static int iy = 0;                   /* Initial Y position of the icon. */

static void check_X_events();
static void process_expose(XExposeEvent *);
static void set_cursor(enum cur_type);

extern char geometry[];       /* X11 geometry information. */
extern char x11_display[];    /* X11 display information. */
extern int iconic;            /* Set if the window is in an iconic state. */
extern unsigned char *mptr;   /* Pointer to scanline data. */

/*  These are the exportable routines used by the popi program.
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c,l)   - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(l,y,w,c)    - to draw an image scanline triple.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype, pos) - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi image window (width, height).
 *  disp_colormap(n,r,g,b)   - load new colormap.
 */


void
disp_init(int argc, char *argv[])        /* Called from main at the start. */
{
    unsigned int h, w;       /* Window dimensions. */
    int flags;
    int x, y;                /* Window position. */
    XSetWindowAttributes winattrs;

    if ((dpy = XOpenDisplay(x11_display)) == NULL) {
        FPRINTF(stderr,"%s: Couldn't open display %s\n", ProgName,
                (getenv ("DISPLAY") ? getenv("DISPLAY") : x11_display));
        exit(1);
    } 

    screen = DefaultScreen(dpy);
    root   = RootWindow(dpy, screen);
    xfd    = ConnectionNumber(dpy);

/* Setup the file descriptors for X server connection and standard input. */

    pfd[0].fd     = xfd;
    pfd[1].fd     = 0;
    pfd[0].events = pfd[1].events = POLLIN;
    npfd          = 2L;

    if (!geometry) {
        STRCPY(geometry, XGetDefault(dpy, ProgName, "Geometry"));
    }

    scr_depth = DefaultDepth(dpy, screen);

    init_dither();            /* Initialise dither arrays and variables. */

    iscolor = (scr_depth > 2);
    if (dtype == IS_MONO) {
        iscolor = 0;
    }
    if (!iscolor) {
        dtype = IS_MONO;
    }

    foregnd   = BlackPixel(dpy, screen);
    backgnd   = WhitePixel(dpy, screen);
    popi_icon = load_icon(icon_image);

    rev_video = (scr_depth == 1 && BlackPixel(dpy, screen) == 0);
    size.flags = PMinSize | PMaxSize | PPosition | PSize;
    size.x = 0;
    size.y = 0;
    size.max_width  = size.min_width  = size.width  = Xsize;
    size.max_height = size.min_height = size.height = Ysize;

    if (strlen(geometry)) {
        flags = XParseGeometry(geometry, &x, &y, &w, &h);
        if (XValue & flags) {
            if (XNegative & flags) {
                x = DisplayWidth(dpy, screen) + x - size.width;
            }
            size.flags |= USPosition;
            size.x = x;
        }
        if (YValue & flags) {
            if (YNegative & flags) {
                y = DisplayHeight(dpy, screen) + y - size.height;
            }
            size.flags |= USPosition;
            size.y = y;
        }
    }    

    winattrs.background_pixel = backgnd;
    winattrs.border_pixel     = backgnd;
    winattrs.backing_store    = NotUseful;
    winattrs.bit_gravity      = NorthWestGravity;
    winattrs.save_under       = False;
    winattrs.event_mask       = FRAME_MASK;

    frame = XCreateWindow(dpy, root,
                          size.x, size.y, size.width, size.height,
                          POPI_BORDER_WIDTH, CopyFromParent,
                          InputOutput, CopyFromParent,
                          CWBackPixel   | CWBackingStore | CWBitGravity |
                          CWBorderPixel | CWEventMask    | CWSaveUnder,
                          &winattrs);

    memarea = XCreatePixmap(dpy, frame, size.width, size.height, scr_depth);

    protocol_atom = XInternAtom(dpy, "WM_PROTOCOLS", False);
    kill_atom = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

    XSetWMProtocols(dpy, frame, &kill_atom, 1);
    XSetStandardProperties(dpy, frame, "popi", NULL, popi_icon,
                           argv, argc, &size);

    wm_hints.icon_x      = ix;
    wm_hints.icon_y      = iy;
    wm_hints.input       = True;
    wm_hints.icon_pixmap = popi_icon;
    wm_hints.flags       = IconPositionHint | InputHint | IconPixmapHint;
    if (iconic) {
        wm_hints.initial_state = IconicState;
        wm_hints.flags |= StateHint;
    }
    XSetWMHints(dpy, frame, &wm_hints);

    gc_mask = GCFunction | GCForeground | GCBackground | GCGraphicsExposures;
    gc_val.foreground = foregnd;
    gc_val.background = backgnd;
    gc_val.function   = GXcopy;
    gc_val.graphics_exposures = False;
    gc = XCreateGC(dpy, root, gc_mask, &gc_val);

    pix_gc = DefaultGC(dpy, screen);

    main_cursor = XCreateFontCursor(dpy, XC_top_left_arrow);
    busy_cursor = XCreateFontCursor(dpy, XC_watch);

    gc_val.foreground = backgnd;
    gc_val.background = foregnd;
    XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val);
    XFillRectangle(dpy, memarea, gc, 0, 0,
                   (unsigned int) size.width, (unsigned int) size.height);

    XSelectInput(dpy, frame, FRAME_MASK);
    XMapWindow(dpy, frame);
    XSync(dpy, 0);
}


void
disp_finish()      /* Called from main prior to exit - null routine. */
{
}


/* Called prior to drawing an image. */

/*ARGSUSED*/
void
disp_imgstart(int width, int height, int ncolors, int len)
{
    PRINTF("Displaying image.\n");
    XClearWindow(dpy, frame);
}


void
disp_imgend()      /* Called after drawing an image - null routine. */
{
}


/* Draw an image scanline triple. */

void
disp_putline(pixel_t **lines, int y, int width, int ncolors)
{
    XImage *image;
    int color, i, j, len, off;
    unsigned char *line, tmp;

    if (scr_depth == 32 || scr_depth == 24) {
        len = width * 4;
    } else if (scr_depth == 8) {
        len = width;
    } else {
        len = (width / 8) + 1;
    }
    mptr = (unsigned char *) Emalloc(len);
    if ((dtype == IS_GRAY || dtype == IS_MONO) && ncolors == 3) {
        line = ntsc_luma(lines, y, width);
    } else {
        line = &lines[0][y*width];
    }

    if (scr_depth == 32 || scr_depth == 24) {
        off = y * width;
        if (colors == 1) {
            for (i = j = 0; i < width; i++) {
                tmp       = lines[0][off+i];
                mptr[j++] = 0;                           /* X */
                mptr[j++] = tmp;                         /* B */
                mptr[j++] = tmp;                         /* G */
                mptr[j++] = tmp;                         /* R */
            }
        } else {
            for (i = j = 0; i < width; i++) {
                mptr[j++] = 0;                         /* X */
                for (color = 2; color >= 0; color--) {
                    mptr[j++] = lines[color][off+i];     /* G, B, R */
                }
            }
        }
    } else if (iscolor) {
        for (i = 0; i < len; i++) {
            mptr[i] = palette[line[i]];
        }
    } else {
        halftone(line, y, width);
    }
 
    image = XCreateImage(dpy, DefaultVisual(dpy, screen), scr_depth,
                         (iscolor) ? ZPixmap : XYPixmap,
                         0, (char *) mptr, width, 1, 8, len);
    XPutImage(dpy, memarea, pix_gc, image, 0, 0, 0, y,
              (unsigned) image->width, (unsigned) image->height);
    XDestroyImage(image);
    XCopyArea(dpy, memarea, frame, gc, 0, y, width, 1, 0, y);
    FREE(mptr);
}


int
disp_getchar()       /* Get next user typed character. */
{
    char c;
    int fd;

    XSync(dpy, 0);
    for (;;) {
        fd = -1 ;
        POLL(pfd, npfd, -1);
        if (pfd[0].revents == POLLIN) {
            fd = POPI_INPUT;
        } else if (pfd[1].revents == POLLIN) {
            fd = KEY_INPUT;
        }

        switch (fd) {     
            case POPI_INPUT : 
                check_X_events();
                break;

            case KEY_INPUT : 
                READ(0, &c, 1);
                if (c == RETURN || c == LINEFEED) {
                    set_cursor(BUSY_CUR);
                }
                return(c);
        }
    }
}


void
disp_percentdone(int percent)
{
    static int lastpercent = 100;

    if (percent == 100) {
        PRINTF("\r    \n");
        FFLUSH(stdout);
        return;
    }
    if (percent != lastpercent && percent % 5 == 0) {
        PRINTF("\r%2d%% ", percent);
        FFLUSH(stdout);
        lastpercent = percent;
    }
}


void
disp_resize(int width, int height)           /* Resize popi image window. */
{
    XSizeHints hints;
    long supplied;

    XResizeWindow(dpy, frame, width, height);
    XGetWMNormalHints(dpy, frame, &hints, &supplied);
    hints.flags      = PMinSize | PMaxSize;
    hints.max_width  = hints.min_width  = width;
    hints.max_height = hints.min_height = height;
    XSetWMNormalHints(dpy, frame, &hints);

    XFreePixmap(dpy, memarea);
    memarea = XCreatePixmap(dpy, frame, width, height, scr_depth);
    gc_val.foreground = backgnd;
    gc_val.background = foregnd;
    XChangeGC(dpy, gc, GCBackground | GCForeground, &gc_val);
    XFillRectangle(dpy, memarea, gc, 0, 0,
                   (unsigned int) width, (unsigned int) height);
}


/* Load new colormap. */

void
disp_colormap(int cmaplen, unsigned char *red, unsigned char *green, 
              unsigned char *blue)
{
    int i, newmap;
    Visual *visual;
    XSetWindowAttributes winattrs;
 
    if (DisplayCells(dpy, screen) <= 2) {
        return;
    }
    visual = DefaultVisual(dpy, screen);
    if (colsused) {
        XFreeColors(dpy, cmap, palette, colsused, 0);
    }
    if (cmap) {
        XFreeColormap(dpy, cmap);
    }

    newmap = 0;
    cmap = DefaultColormap(dpy, screen);

/*  Attempt to use the default colormap. If we can't allocate all the colors
 *  we need, then attempt to create a private colormap.
 */

    ccol.flags = DoRed | DoGreen | DoBlue;
    iscolor = 1;
    for (i = 0; i < CMAPLEN; i++) {
        ccol.red   = red[i]   << 8;
        ccol.green = green[i] << 8;
        ccol.blue  = blue[i]  << 8;
        if (!XAllocColor(dpy, cmap, &ccol)) {
            if ((visual->class == StaticColor) ||
                (visual->class == StaticGray)) {
                FPRINTF(stderr, 
                        "popi: XAllocColor failed on a static visual\n");
                iscolor = 0;
                return;
            } else { 

/*  We can't allocate the colors shareable so free all the colors we had
 *  allocated and create a private colormap.
 */

                XFreeColors(dpy, cmap, palette, i, 0);
                newmap = 1;
                break;
            }
        }
        palette[i] = ccol.pixel;
    }

    if (newmap) {
        if ((visual->class == PseudoColor) || (visual->class == GrayScale)) {
            cmap = XCreateColormap(dpy, root, visual, AllocNone);
        } else {
            cmap = XCreateColormap(dpy, root, visual, AllocAll);
        }
 
        if (!XAllocColorCells(dpy, cmap, False, NULL, 0, palette, cmaplen)) {
            FPRINTF(stderr, "popi: failed to allocate private colormap.\n");
            iscolor = 0;
            return;
        } 
 
        ccol.flags = DoRed | DoGreen | DoBlue;
        for (i = 0; i < cmaplen; i++) {
            ccol.pixel = palette[i];
            ccol.red   = red[i]   << 8;
            ccol.green = green[i] << 8;
            ccol.blue  = blue[i]  << 8;
            XStoreColor(dpy, cmap, &ccol);
        } 
    }
 
    winattrs.colormap = cmap;
    XChangeWindowAttributes(dpy, frame, CWColormap, &winattrs);
}


int
disp_prompt()        /* Display popi prompt and clear input line. */
{
    static char prompt[] = "-> ";

    set_cursor(NORMAL_CUR);
    PRINTF(prompt);
    FFLUSH(stdout);

    return(sizeof(prompt-1));
}


void
disp_error(int errtype, int pos)    /* Display error message. */
{
    extern int errno;
    extern char *sys_errlist[];

    if (errtype & ERR_PARSE) {
        int i;

        for (i = 1; i < pos; ++i) {
            PUTC('-', stderr);
        }
        PUTC('^', stderr);
        PUTC('\n', stderr);
    }

    FPRINTF(stderr, "%s\n", ErrBuf);

/* We assume errno hasn't been reset by the preceding output. */

    if (errtype & ERR_SYS) {
        FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]);
    }
    FFLUSH(stderr);
}


static void
check_X_events()      /* Check and process pending X events. */
{
    XClientMessageEvent *ev;

    XSync(dpy, 0);
    while (XPending(dpy) && (XPeekEvent(dpy, &event), (event.type == Expose))) {
        XNextEvent(dpy, &event);

        switch (event.type) {
            case ClientMessage : 
                /* Catch ICCCM kill from WM. */

                ev = (XClientMessageEvent *) &event;
                if (ev->message_type == protocol_atom &&
                    ev->data.l[0] == kill_atom) {
                    exit(0);
                }
                break;

            case Expose : 
                process_expose((XExposeEvent *) &event);
        }
    }
}


static Pixmap
load_icon(unsigned short sbuf[])
{
    unsigned char cbuf[512];
    int i;
    GC igc;
    Pixmap pixmap;
    XImage *image;

    for (i = 0; i < 256; i++) {
        cbuf[i*2  ] = (sbuf[i] >> 8) & 0xFF;
        cbuf[i*2+1] =  sbuf[i]       & 0xFF;
    }
    pixmap = XCreatePixmap(dpy, root, 64, 64, 1);
    gc_mask = GCFunction | GCForeground | GCBackground | GCGraphicsExposures;
    gc_val.foreground         = foregnd;
    gc_val.background         = backgnd;
    gc_val.function           = GXcopy;
    gc_val.graphics_exposures = False;
    igc   = XCreateGC(dpy, pixmap, gc_mask, &gc_val);
    image = XCreateImage(dpy, DefaultVisual(dpy, screen), 1, XYPixmap,
                         0, (char *) cbuf, 64, 64, BitmapPad(dpy), 0);
    XPutImage(dpy, pixmap, igc, image, 0, 0, 0, 0, 64, 64);

    return(pixmap);
}


static void
process_expose(XExposeEvent *event)
{
    int doframe = 0;

    do {
        if (event->count == 0) {
            if (event->window == frame) {
                doframe++;
            }
        }
    } while (XCheckMaskEvent(dpy, ExposureMask, (XEvent *) event));

    if (doframe) {
        XCopyArea(dpy, memarea, frame, gc, 0, 0, Xsize, Ysize, 0, 0);
    }
}


static void
set_cursor(enum cur_type type)
{
    switch (type) {
        case BUSY_CUR : 
            XDefineCursor(dpy, frame, busy_cursor);
            break;
        case NORMAL_CUR : 
            XDefineCursor(dpy, frame, main_cursor);
    }
}
