
/*  @(#)trig.c 1.3 02/02/15
 *
 *  Various trig and builtin functions used by popi.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

#include <unistd.h>
#include "popi.h"

#define PI           (3.141592653589793238)
#define PI_OVER_MAX2 (PI / (MAX_ANG / 2.0))
#define MA_0         (0)
#define MA_90        (MAX_ANG / 4)
#define MA_180       (MAX_ANG / 2)
#define MA_270       (MAX_ANG * 3 / 4)
#define MA_360       (MAX_ANG)

#include <math.h>

int sin_cache[MAX_ANG+5], cos_cache[MAX_ANG+5], tan_cache[MAX_ANG+5];

static int atan_cache[1030];


void
cache_trig()
{
    int i;

    for (i = 0; i < MAX_ANG+5; i++) {
        sin_cache[i] = MAX_TRIG * sin(i * PI_OVER_MAX2);
        cos_cache[i] = MAX_TRIG * cos(i * PI_OVER_MAX2);
        tan_cache[i] = MAX_TRIG * tan(i * PI_OVER_MAX2);
    }
    for (i = 0; i < 1030; i++) {
        atan_cache[i] = atan(i / 1024.0) * MA_180 / PI;
    }

    srand(getpid());
}


int
isqrt(int i)
{
    return((int) sqrt(1.0 * i));
}


#define abs(x) (x < 0 ? -(x) : x)


int
dist(int x, int y, int x2, int y2)
{
    float dx = abs(x-x2);
    float dy = abs(y-y2);

    return((int) sqrt(dx*dx + dy*dy));
}


static struct {
    int adj, scale;
} octs[8] = {
    { MA_0,    1 },
    { MA_90,  -1 },
    { MA_180, -1 },
    { MA_90,   1 },
    { MA_360, -1 },
    { MA_270,  1 },
    { MA_180,  1 },
    { MA_270, -1 }
};

 
int
angle(int x, int y, int x2, int y2)
{
    int idx = 0, ang;
    int dx = x - x2;
    int dy = y2 - y;

    if (dx == 0 && dy == 0) {
        return(0);
    }
         
    if (dx < 0) {
        dx = -dx;
        idx += 2;
    }
    if (dy < 0) {
        dy = -dy;
        idx += 4;
    }
    if (dx < dy) {
        int tmp = dx;

        dx = dy;
        dy = tmp;
        idx++;
    }
    ang = atan_cache[(dy * 1024) / dx];

    return(octs[idx].adj + (octs[idx].scale == -1 ? -ang : ang));
}


#define clamp(v, l, u)  ((v) < (l) ? (l) : (v) > (u) ? (u) : (v))


int
polar(int a, int r)
{
    int x, y;

    if (abs(a) >= MAX_ANG) {
        a %= MAX_ANG;
    }
    if (a < 0) {
        a += MAX_ANG;
    }

    x = cos_cache[a] * r / MAX_TRIG + Xsize / 2;
    y = Ysize - (sin_cache[a] * r / MAX_TRIG + Ysize / 2);

    x = clamp(x, 0, Xsize-1);
    y = clamp(y, 0, Ysize-1);

    return(y * Ysize + x);
}


int
dorand()
{
    return(rand() % MAX_TRIG);
}


/*ARGSUSED*/
int
doatan(int x, int y)
{
    FPRINTF(stderr, "doatan entered. Not implemented.\n");

    return(0);
}


/*ARGSUSED*/
int
dohypot(int x, int y)
{
    FPRINTF(stderr, "dohypot entered. Not implemented.\n");

    return(0);
}
