
/*  @(#)run.c 1.3 02/02/15
 *
 *  Run time interpreter used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

#include <ctype.h>
#include "popi.h"
#include <math.h>

#define STACKSIZE  128

#define dop(OP)    a = *--sp; tp = sp-1; *tp = (*tp OP (pstack_t) a)
#if defined (ISTACK)
#define  diop(OP)   dop(OP)
#else
#define diop(OP)   ca = *--sp; tp = sp-1; cb = *tp; *tp = (cb OP ca)
#endif /*ISTACK*/

/* Local function prototypes */

static void prun(void);


void
SwapOldNew()
{
    struct SRC *tmp;

    tmp = CurOld;
    CurOld = CurNew;
    CurNew = tmp;
    STRCPY(CurOld->str, "old");
    STRCPY(CurNew->str, "new");
}


static void
prun()
{
    long *ParseEnd;     /* After end of parse string */
    int color;
    int x, y;           /* Coordinates */
    int nrange = 0;     /* No. range errors */

    ParseEnd = &parsed[prs];

#ifndef NDEBUG
    if (Debug) {
        long *CurrParse;  /* Pointer to current item in parse string */

        FPRINTF(Debug, "Parse string\n");
        for (CurrParse = parsed; CurrParse != ParseEnd; ++CurrParse)
            FPRINTF(Debug, "'%c' (%ld)\n", (char) *CurrParse, *CurrParse);
        FPRINTF(Debug, "---\n");
    }
#endif /* NDEBUG */
    DEBUG((Debug, "computing %i colors\n", col_used));

/*  Warning: Microsoft Quick C generates incorrect code for the following
 *  loop when optimisation is turned on.
 */

    for (y = 0; y < CurNew->height; ++y) {
#if defined(ISTACK)
        short *ap;    /* Precalculated polar angles */
        short *rp;    /* Precalculated polar radius */
#else
        float *ap;
        float *rp;
#endif /*ISTACK*/
        pixel_t *pixp;            /* Default destination. */
        pixel_t *pixaddr = NULL;  /* Explicit destination. */
        union IntFloat intfloat;  /* Get floats from array of long. */

        ap = &avals[y * CurNew->width];
        rp = &rvals[y * CurNew->width];

        for (color = 0; color < col_used; ++color) {
            pixp = &CurNew->pix[color][y*Xsize];

            for (x = 0; x < CurNew->width; ++x, ++pixp) {
                pstack_t Stack[STACKSIZE];   /* The stack */
                pstack_t *sp;                /* Stack pointer (top of stack) */
                pstack_t *tp;                /* Temp stack pointer */
                pstack_t a, b;
                int c;                       /* Scratch */
                long ca, cb;
                register long *CurrParse;    /* Pointer to current item */
                                             /* in parse string */
                double dt;             /* overflow tester */

                for (CurrParse = parsed, sp = Stack; CurrParse != ParseEnd;
                     ++CurrParse) {
                    switch ((int) *CurrParse) {
                        case OP_VALUE : 
                            *sp++ = (pstack_t) *++CurrParse;
                            continue;

                        case OP_VALF  : 
                            intfloat.intgr = *++CurrParse;
                            *sp++ = (pstack_t) intfloat.flt;
                            continue;

                        case OP_AT : 
                            --sp;
                            if (Truncate) {
                                if (*sp > (pstack_t) Zmax) {
                                    *sp = Zmax;
                                } else if (*sp < 0) {
                                    *sp = 0;
                                }
                            }
                            *pixp = (pixel_t) *sp;
                            continue;

                        case OP_PLUS : 
                            dop(+);
                            break;

                        case OP_MINUS : 
                            dop(-);
                            break;

                        case OP_MUL : 
                            dop(*);
                            break;

                        case OP_DIV : 
                            a = *--sp;
                            tp = sp-1;
                            if (a == 0) {
                                *tp = Zmax;
                            } else {
                                *tp = (*tp / (pstack_t) a);
                            }
                            break;

                        case OP_MOD : 
                            ca = *--sp;
                            tp = sp-1;
                            if (ca == 0) {
                                *tp = 0;
                            } else {
                                cb = *tp;
                                *tp = (cb % ca);
                            }
                            break;

                        case OP_GT : 
                            dop(>);
                            break;

                        case OP_LT : 
                            dop(<);
                            break;

                        case OP_GE : 
                            dop(>=);
                            break;

                        case OP_LE : 
                            dop(<=);
                            break;

                        case OP_EQ : 
                            dop(==);
                            break;

                        case OP_NE : 
                            dop(!=);
                            break;

                        case OP_CAND : 
                            diop(&&);
                            break;

                        case OP_COR : 
                            diop(||);
                            break;

                        case OP_XOR : 
                            diop(^);
                            break;

                        case OP_OR : 
                            diop(|);
                            break;

                        case OP_AND : 
                            diop(&);
                            break;

                        case OP_X : 
                            *sp++ = (pstack_t) x;
                            break;

                        case OP_Y : 
                            *sp++ = (pstack_t) y;
                            break;

                        case OP_UMIN : 
                            tp = sp-1;
                            *tp = -(*tp);
                            break;

                        case OP_NOT : 
                            tp = sp-1;
                            *tp = !(*tp);
                            break;

                        case OP_ASGN : 
                            a = *--sp;
                            --sp;
                            if (Truncate) {
                                if (a > (pstack_t) Zmax) {
                                    a = Zmax;
                                } else if (a < 0) {
                                    a = 0;
                                }
                            }
                            if (pixaddr) {
                                *pixaddr = (pixel_t) a;
                            }
                            break;

                        case OP_CRVAL : 
                            ca = *--sp;              /* y */
                            cb = *--sp;              /* x */
                            tp = sp-1;
                            c = (int) *tp;
                            if (ca >= Images[c].height  || ca < 0 ||
                                cb >= Images[c].width   || cb < 0) {
                                if (RangeCheck && (nrange++ == 0)) {
                                    FPRINTF(stderr,
                                      "Range err at (%d,%d) => %s[%ld, %ld]\n",
                                      x, y, Images[c].str, cb, ca);
                                }
                                switch (rng_policy) {
                                    case CUT :
                                        *tp = (pstack_t) 0;
                                        break;

                                    case WRAP :
                                        if (ca >= Images[c].height) {
                                            ca = ca % Images[c].height;
                                        }
                                        if (ca < 0) {
                                            ca = Images[c].height + 
                                                 (++ca % Images[c].height) - 1;
                                        }
                                        if (cb >= Images[c].width) {
                                            cb = cb % Images[c].width;
                                        }
                                        if (cb < 0) {
                                            cb = Images[c].width + 
                                                 (++cb % Images[c].width) - 1;
                                        }
                                        *tp = (pstack_t) 
                                        Images[c].pix[color][(ca*Images[c].Xalloc)+cb];
                                        break;

                                    case MINMAX :
                                        if (ca >= Images[c].height) {
                                            ca = Images[c].height - 1;
                                        }
                                        if (ca < 0) {
                                            ca = 0;
                                        }
                                        if (cb >= Images[c].width) {
                                            cb = Images[c].width - 1;
                                        }
                                        if (cb < 0) {
                                            cb = 0;
                                        }
                                        *tp = (pstack_t) 
                                        Images[c].pix[color][(ca*Images[c].Xalloc)+cb];
                                        break;
                                }
                            } else {
                                *tp = (pstack_t) 
                                 Images[c].pix[color][(ca*Images[c].Xalloc)+cb];
                            }
                            break;

                        case OP_CLVAL : 
                            ca = *--sp;              /* y */
                            cb = *--sp;              /* x */
                            tp = sp-1;
                            c = (int) *tp;
                            if (ca >= Images[c].height  || ca < 0 ||
                                cb >= Images[c].width   || cb < 0) {
                                if (RangeCheck && (nrange++ == 0)) {
                                    FPRINTF(stderr,
                                      "Range err at (%d,%d) => %s[%ld, %ld]\n",
                                      x, y, Images[c].str, cb, ca);
                                }
                                switch (rng_policy) {
                                    case CUT :
                                    case MINMAX :
                                        pixaddr = NULL;
                                        break;

                                    case WRAP :
                                        if (ca >= Images[c].height) {
                                            ca = ca % Images[c].height;
                                        }
                                        if (ca < 0) {
                                            ca = Images[c].height + 
                                                 (++ca % Images[c].height) - 1;
                                        }
                                        if (cb >= Images[c].width) {
                                            cb = cb % Images[c].width;
                                        }
                                        if (cb < 0) {
                                            cb = Images[c].width + 
                                                 (++cb % Images[c].width) - 1;
                                        }
                                        pixaddr = &(Images[c].pix[color][(ca*Images[c].Xalloc)+cb]);
                                        break;
                                }
                            } else {
                                pixaddr = &(Images[c].pix[color][(ca*Images[c].Xalloc)+cb]);
                            }
                            break;

                        case OP_PRVAL : 
                            {
                                int xval, yval;
    
                                a = *--sp;    /* Angle */
                                b = *--sp;    /* Radius */

                                tp = sp - 1;
                                c = (int) *tp;

                                xval = (int) (b * cos((double) DtoR(a)) +
                                              Images[c].width / 2.0);
                                yval = (int) (- b * sin((double) DtoR(a)) +
                                              Images[c].height / 2.0);

                                if (yval >= Images[c].height  || yval < 0 ||
                                    xval >= Images[c].width   || xval < 0) {
                                    if (RangeCheck && (nrange++ == 0)) {
                                        FPRINTF(stderr,
                                            "Range err at (%d,%d) => %s{%d,%d} [%d,%d]\n",
                                            x, y, Images[c].str, b, a, xval, yval);
                                    }
                                    switch (rng_policy) {
                                        case CUT :
                                            *tp = (pstack_t) 0;
                                            break;

                                        case WRAP :
                                            if (yval >= Images[c].height) {
                                                yval = yval % Images[c].height;
                                            }
                                            if (yval < 0) {
                                                yval = Images[c].height + (++yval % Images[c].height) - 1;
                                            }
                                            if (xval >= Images[c].width) {
                                                xval = xval % Images[c].width;
                                            }
                                            if (xval < 0) {
                                                xval = Images[c].width + (++xval % Images[c].width) - 1;
                                            }
                                            *tp = (pstack_t) 
                                                  Images[c].pix[color][(yval*Images[c].Xalloc)+xval];
                                            break;

                                        case MINMAX :
                                            if (yval >= Images[c].height) {
                                                yval = Images[c].height - 1;
                                            }
                                            if (yval < 0) {
                                                yval = 0;
                                            }
                                            if (xval >= Images[c].width) {
                                                xval = Images[c].width - 1;
                                            }
                                            if (xval < 0) {
                                                xval = 0;
                                            }
                                            *tp = (pstack_t) 
                                                Images[c].pix[color][(yval*Images[c].Xalloc)+xval];
                                            break;
                                    }
                                } else {
                                    *tp = (pstack_t) 
                                        Images[c].pix[color][(yval*Images[c].Xalloc)+xval];
                                }
                            }
                            break;

                        case OP_PLVAL : 
                            {
                                int xval, yval;

                                a = *--sp;    /* Angle */
                                b = *--sp;    /* Radius */

                                tp = sp - 1;
                                c = (int) *tp;

                                xval = (int) (b * cos((double) DtoR(a)) +
                                              Images[c].width / 2.0);
                                yval = (int) (- b * sin((double) DtoR(a)) +
                                              Images[c].height / 2.0);

                                if (yval >= Images[c].height  || yval < 0 ||
                                    xval >= Images[c].width   || xval < 0) {
                                    if (RangeCheck && (nrange++ == 0)) {
                                        FPRINTF(stderr,
                                            "Range err at (%d,%d) => %s{%d,%d} [%d,%d]\n",
                                            x, y, Images[c].str, b, a, xval, yval);
                                    }
                                    switch (rng_policy) {
                                        case CUT :
                                        case MINMAX :
                                            pixaddr = NULL;
                                            break;

                                        case WRAP :
                                            if (yval >= Images[c].height) {
                                                yval = yval % Images[c].height;
                                            }
                                            if (yval < 0) {
                                                yval = Images[c].height + (++yval % Images[c].height) - 1;
                                            }
                                            if (xval >= Images[c].width) {
                                                xval = xval % Images[c].width;
                                            }
                                            if (xval < 0) {
                                                xval = Images[c].width + (++xval % Images[c].width) - 1;
                                            }
                                            pixaddr = &(Images[c].pix[color][(yval*Images[c].Xalloc)+xval]);
                                            break;
                                    }
                                } else {
                                    pixaddr = &(Images[c].pix[color][(yval*Images[c].Xalloc)+xval]);
                                }
                            }
                            break;

                        case OP_POW : 
                            a = *--sp;     /* Exponent */
                            tp = sp-1;
                            /* test for conversion overflow */
                            if ( *tp != 0 &&
                               (dt = pow((double) *tp, (double) a)) <=
                               (double) MAXLONG && dt >= -MAXLONG) {
                                *tp = (pstack_t) dt;
                            }    /* else leave *tp as is? */
                            break;

                        case OP_A :
#if NOMEM
                            *sp++ = (pstack_t)
                                    RtoD(atan2((double) y, (double) x)) + 0.5;
#else /* !NOMEM */
                            *sp++ = (pstack_t) *ap;
#endif /* !NOMEM */
                            break;

                        case OP_R : 
                            *sp++ = (pstack_t) *rp;
                            break;

                        case OP_SIN : 
                            tp = sp-1;
                            *tp = (pstack_t) (sin((double) DtoR(*tp))
#if defined(ISTACK)    /* on integer stack scale it by Z */
                                  * (double) Zmax
#endif
                                  );
                            break;

                        case OP_COS : 
                            tp = sp-1;
                            *tp = (pstack_t) (cos((double) DtoR(*tp))
#if defined(ISTACK)     /* on integer stack scale it by Z */
                                  * (double) Zmax
#endif
                                  );
                            break;

                        case OP_ATAN : 
                            a = *--sp;
                            tp = sp-1;
                            *tp = (pstack_t)
                                  RtoD(atan2((double) *tp, (double) a));
                            break;

                        case OP_HYPOT : 
                            a = *--sp;
                            tp = sp - 1;
                            *tp = (pstack_t) hypot((double) *tp, (double) a);
                            break;

                        case OP_ABS : 
                            tp = sp-1;
                            *tp = *tp < 0 ? - *tp : *tp;
                            break;

                        case OP_LOG : 
                            tp = sp-1;
                            if (*tp > 0) {
                                *tp = (pstack_t) log((double) *tp);
                            }
                            break;

                        case OP_SQRT : 
                            tp = sp-1;
                            if (*tp > 0) {
                                *tp = (pstack_t) sqrt((double) *tp);
                            }
                            break;

                        case OP_RAND : 
                            *sp++ = (pstack_t) RANDOM;
                            break;

                        case OP_LSHIFT : 
                            diop(<<);
                            break;

                        case OP_RSHIFT : 
                            diop(>>);
                            break;

                        case OP_CONDIT : 
                            ca = *--sp;
                            CurrParse++;
                            if (!ca) {
                                CurrParse = &parsed[*CurrParse];
                            }
                            break;

                        case OP_COLON : 
                            CurrParse = &parsed[CurrParse[1]];
                            break;

                        default : 
                            if (*CurrParse < 127 && isprint(*CurrParse)) {
                                SPRINTF(ErrBuf, 
                                        "run: unknown operator '%c' (%d)",
                                        *CurrParse, *CurrParse);
                            } else {
                                SPRINTF(ErrBuf, "run: unknown operator %d",
                                        *CurrParse);
                            }
                            error(ERR_SNARK);
                            return;
                    }
                }
                ++ap;
                ++rp;
            }
        }

        disp_percentdone(y * 100 / (Ysize-1));
    }
}


void
run()
{
    prun();
    disp_percentdone(100);  /* Can rely on '100' to terminate */
    CurNew->ncolors = col_used;

    if (disp_active) {
        drawimg(CurNew);
    }

    SwapOldNew();
}


void
drawimg(struct SRC *old)
{
    int i, y;
    unsigned char red[CMAPLEN], green[CMAPLEN], blue[CMAPLEN];
    struct SRC *new;

    disp_imgstart(old->width, old->height, old->ncolors, old->Xalloc);
    if (old->ncolors == 3 && dtype == IS_COLOR) {
        if (scr_depth == 8) {
            new = dither24to8(old);
        } else {
            new = old;
        }
        for (y = 0; y < new->height; y++) {
            disp_putline((pixel_t **) new->pix, y, new->width, new->ncolors);
        }
        if (scr_depth == 8) {
            ImgFree(new);
        }
    } else if (dtype == IS_GRAY || dtype == IS_COLOR) {
        for (i = 0; i < CMAPLEN; i++) {
            red[i] = green[i] = blue[i] = (unsigned char) i;
        }
        disp_colormap(CMAPLEN, red, green, blue);
        for (y = 0; y < old->height; y++) {
            disp_putline((pixel_t **) old->pix, y, old->width, old->ncolors);
        }
    } else {
        for (y = 0; y < old->height; y++) {
            disp_putline((pixel_t **) old->pix, y, old->width, old->ncolors);
        }
    }
    disp_imgend();
}


/*  Convert the three RGB lines into one NTSC luma line, for display
 *  on grayscale devices.  (Just pass pointer along if colors == 1)
 */

pixel_t*
ntsc_luma(pixel_t **lines, int y, int width)
{
    int off, x;

    if (colors == 1) {
        return(lines[0]);
    }

    CLEARMEM((char *) ntsc, width * (sizeof(pixel_t)));
    off = y * width;
    for (x = 0; x < width; x++) {
        ntsc[x] = lines[0][off+x] * 0.299 +       /* R */
                  lines[1][off+x] * 0.587 +       /* G */
                  lines[2][off+x] * 0.114;        /* B */
    }

    return(ntsc);
}
