
/*  @(#)polar.c 1.3 02/02/15
 *
 *  Polar coordinate handling routines used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include "popi.h"
#include <math.h>

#if defined(ISTACK)
#define POLARTYPE short
#define ROUNDUP +0.5
#define PFILE "/usr/tmp/popi.%d.%d"
#else
#define POLARTYPE float
#define ROUNDUP
#define PFILE "/usr/tmp/popi_f.%d.%d"
#endif /*ISTACK*/

POLARTYPE *avals = 0;
POLARTYPE *rvals = 0;

#ifdef AMIGA
double
hypot(double x, double y)
{
    return(sqrt(x*x + y*y));
}
#endif  /* AMIGA */


extern int tcache;


void
MakePolar()
{
    char path[30];
    POLARTYPE *ap;     /* Pointer to angle array */
    POLARTYPE *rp;     /* Pointer into radius array */
    POLARTYPE *a1;     /* Tmp pointer to angle array */
    POLARTYPE *r1;     /* Tmp pointer to radius array */
    int fd, n, pct, pctl, size, x, y, xmin, xmax, ymin, ymax;
    double xd = 0;
    double yd;

    if (avals) {
        return;        /* Previously calculated */
    }

    size  = CurNew->width * CurNew->height * sizeof(POLARTYPE);
    avals = (POLARTYPE *) LINT_CAST(Emalloc((unsigned) size));
    rvals = (POLARTYPE *) LINT_CAST(Emalloc((unsigned) size));

    if (tcache) {
        SPRINTF(path, PFILE, CurNew->width, CurNew->height);
        if ((fd = open(path, O_RDONLY)) != -1) {
            n  = read(fd, (POLARTYPE *) avals, size);
            n += read(fd, (POLARTYPE *) rvals, size);
            CLOSE(fd);
            if (n == 2*size) {
                return;
            }
        } else {
            perror("polar open (read) error");
        }
    }

    ymax =  CurNew->height / 2;
    ymin = -(CurNew->height - 1) / 2;
    xmin = -CurNew->width / 2;
    xmax =  (CurNew->width - 1) / 2;

    rp = rvals;
    ap = avals;

    FPRINTF(stderr, "Calculating trig tables\n");
    FFLUSH(stderr);

/*        +y
 *         ^
 *       2 | 1
 * -x <----+----> +x
 *       3 | 4
 *         v
 *        -y
 */

    pctl = -1;
    for (y = ymax; y >= 0; --y) {
        pct = ((ymax-y) * 100) / ymax;
        if (pct != pctl) {
            disp_percentdone(pctl=pct);
        }
        yd = y;

/* Quadrant 2 */

        for (x = xmin; x < 0; ++x) {
            xd = x;
            *ap++ = (POLARTYPE) (RtoD(atan2(yd, xd)) ROUNDUP);
            *rp++ = (POLARTYPE) hypot(yd, xd);
        }
        *ap++ = (y == 0) ? 0 : (POLARTYPE) (RtoD(atan2(yd, xd)) ROUNDUP);
        *rp++ = (y == 0) ? 0 : (POLARTYPE) hypot(yd, xd);
        ++x;

/* Remember location just before the 0 value */

        r1 = rp - 2;
        a1 = ap - 2;

/* Quadrant 1 */

        for (; x <= xmax; ++x) {
            *ap++ = 180 - *a1--;
            *rp++ =       *r1--;
        }
    }


/* Quadrant 3, 4 */

    r1 = rp - CurNew->width - 1;
    a1 = ap - CurNew->width - 1;
    for (; y >= ymin; --y) {
        for (x = xmin; x <= xmax; ++x) {
            *rp++ = *r1--;
            *ap++ = *a1-- + 180;
        }
    }
    disp_percentdone(100);      /* Can rely on '100' to terminate */
 
    if (tcache) {
        if ((fd = open(path, O_WRONLY|O_CREAT, 0777)) != -1) {
            n  = write(fd, (POLARTYPE *) avals, size);
            n += write(fd, (POLARTYPE *) rvals, size);
            CLOSE(fd);
            if (n == 2*size) {
                return;
            }
        } else {
            perror("polar open (write) error");
        }
    }
}
