
/*  @(#)mk.c 1.3 02/02/15
 *
 *  Tree node creation routines used in the popi yacc grammar.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

#include <stdarg.h>
#include "popi.h"
#include "expr.h"


Tree *
mk(enum tree t, Tree *a, Tree *b)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(2));
    new->kids[0] = (void *) a;
    new->kids[1] = (void *) b;
    new->t = t;

    return(new);
}


Tree *
treealloc(int x)
{
    Tree *t;

    t = alloc(offsetof(Tree, kids[x]));
    t->siz = x;

    return(t);
}


Tree *
mkInum(int i)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(0));

    new->i = i;
    new->t = T_Inum;

    return(new);
}


Tree *
mkFnum(double d)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(0));

    new->d = d;
    new->t = T_Fnum;

    return(new);
}


Tree *
mkCoord(int file, Tree *coord)
{
    Tree *new = LINT_CAST(treealloc(1));

    new->i = file;
    if ((new->kids[0] = coord) == 0) {
        new->siz = 0;
    }
    new->t = T_Coord;

    return(new);
}

 
Tree *
mkMapcoord(int file, Tree *coord, Tree *expr)
{
    Tree *new = LINT_CAST(treealloc(2));

    new->i = file;
    if ((new->kids[1] = coord) == 0) {
        new->siz--;
    }
    new->t = T_Mapcoord;
    new->kids[0] = expr;

    return(new);
}


Tree *
mkun(enum tree t, void *a)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(1));

    new->kids[0] = a;
    new->t = t;

    return(new);
}


Tree *
mkCond(Tree *a, Tree *b, Tree *c)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(3));

    new->kids[0] = (void *) a;
    new->kids[1] = (void *) b;
    new->kids[2] = (void *) c;
    new->t = T_Cond;

    return(new);
}


Tree *
mkempty(enum tree t)
{
    Tree *new = (Tree *) LINT_CAST(treealloc(0));

    new->t = t;

    return(new);
}


struct builtin builtins[] = {
    { "sin",   T_Sin,     1, NULL   },
    { "cos",   T_Cos,     1, NULL   },
    { "tan",   T_Tan,     1, NULL   },
    { "abs",   T_Abs,     1, NULL   },
    { "sqr",   T_Sqr,     1, NULL   },
    { "sqrt",  T_Sqrt,    1, isqrt  },
    { "rand",  T_Rand,    0, dorand },
    { "atan",  T_Atan,    2, NULL   },
    { "hypot", T_Hypot,   2, NULL   },
    { "log",   T_Log,     1, NULL   },
    { "dist",  T_VRadius, 4, dist   },
    { "angle", T_VAngle,  4, angle  },
};

int nbuiltin = NUMEL(builtins);


Tree *
mkBuiltin(char *f, int n, ...)
{
    int i;
    Tree *new;
    va_list va;

    for (i = 0; i < nbuiltin; i++) {
        if (!strcmp(f, builtins[i].n)) {
            break;
        }
    }

    if (i == nbuiltin) {
        FPRINTF(stderr, "No such builtin\n");
        return(NULL);
    }
    if (builtins[i].a != n) {
        FPRINTF(stderr, "Wrong number of args for %s()\n", f);
        return(NULL);
    }
    new = treealloc(n);
    new->t = builtins[i].t;
    va_start(va, n);
    for (i = 0; i < n; i++) {
        new->kids[i] = va_arg(va, Tree *);
    }
    va_end(va);

    return(new);
}


Tree *
mkAssign(char *name, Tree *val)
{
    Tree *t = treealloc(1);

    t->t = T_Assign;
    t->i = allocName(name);
    t->kids[0] = val;

    return(t);
}


Tree *
mkVar(char *n)
{
    Tree *t = treealloc(0);

    t->t = T_Var;
    t->i = allocName(n);

    return(t);
}


Tree *
mkTop(Tree *t1, Tree *t2, Tree *t3, Tree *t4, Tree *t5)
{
    Tree *new = treealloc(5);

    new->t = T_Top;
    new->kids[0] = t1;
    new->kids[1] = t2;
    new->kids[2] = t3;
    new->kids[3] = t4;
    new->kids[4] = t5;

    return(new);
}
