
/*  @(#)expr.y 1.2 02/02/11
 *
 *  Yacc grammar used by the popi program.
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Copyright (c) 1989-2002 - Rich Burridge, Sun Microsystems Inc.
 *  All rights reserved.
 *
 *  Permission is given to distribute these sources, as long as the
 *  copyright messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if
 *  reported to me, then an attempt will be made to fix them.
 */

%{

#include <stdio.h>
#include <ctype.h>
#include <values.h>
#include <sys/param.h>
#include "popi.h"
#include "expr.h"

#define RET(x)    return(emit((x)))
#define STRDUP(x) strcpy(alloc(strlen(x)+1), x)

enum word_type { W_Special, W_Function, W_Keyword };

static int check_buf(void);
static int emit(int);
static int Getch(void);
static int GetNumber(int);
static int is_word(enum word_type, char *);
static int parse_default(int);
static int yylex(void);
int yydebug = 1;

static void pushback(int);
static void yyerror(char *);

struct NameToken {
    char *name;
    int token;
};

static double lastfnum;       /* Last floating point number parsed. */
static int lastinum;          /* Last integer number parsed. */

static char buf[MAXLINE];

%}

%union {
    Tree *node;
    char *name;
    int num;
}

%token TOK_ABS     TOK_ANGLE   TOK_AND      TOK_ATAN
%token TOK_BLUE    TOK_COLOR   TOK_COMB     TOK_COMMENT
%token TOK_COS     TOK_DEBUG   TOK_DISPLAY  TOK_DIST
%token TOK_EQ      TOK_EXPLO   TOK_FCON     TOK_FNAME
%token TOK_FREE    TOK_GE      TOK_GENEPSON TOK_GENPS
%token TOK_GET     TOK_GREEN   TOK_GRAY     TOK_HELP
%token TOK_HYPOT   TOK_ICON    TOK_IMAGE    TOK_LE
%token TOK_LIST    TOK_LOG     TOK_LOGFILE  TOK_LSHIFT
%token TOK_LUMA    TOK_MATTE   TOK_MELT     TOK_MONO
%token TOK_NE      TOK_NEW     TOK_OFMT     TOK_OIL
%token TOK_OR      TOK_POW     TOK_PUT      TOK_QUIT
%token TOK_RAND    TOK_RANGE   TOK_RCUT     TOK_READ
%token TOK_RED     TOK_RMINMAX TOK_RSHIFT   TOK_RWRAP
%token TOK_SHEAR   TOK_SIGNED  TOK_SIN      TOK_SIZE
%token TOK_SLICE   TOK_SQRT    TOK_STORE    TOK_TILE
%token TOK_TRUNC   TOK_UNDO    TOK_VAR      TOK_VERBOSE
%token TOK_VERSION TOK_WRITE

%type <node> popi comment cmd dst mapdst special expr co as ss
%type <node> or xor and eq rel shift plus times pow factor
%type <node> unaryop fncall0 fncall1 fncall2 fncall4 display image
%type <node> dfile coord dir number logfile policy rgbcomp

%type <name> TOK_ABS    TOK_ANGLE TOK_ATAN  TOK_COS  TOK_DIST
%type <name> TOK_FNAME  TOK_HYPOT TOK_IMAGE TOK_LOG  TOK_NEW  TOK_RAND
%type <name> TOK_SIN    TOK_SQRT  TOK_VAR
%type <num>  file

%start popi

%%

popi    : comment               { tree = NULL; YYACCEPT;        }
        | error '\n'            { tree = NULL; YYABORT;         }
        | cmd comment           { tree = $1;   YYACCEPT;        }

comment : /* empty */           { $$ = NULL;                     }
        | TOK_COMMENT '\n'      { $$ = NULL;                     }
        | '\n'                  { $$ = NULL;                     }

cmd     : dst
        | mapdst
        | special
        | 'w' TOK_IMAGE         { $$ = mkun(T_Write, $2);        }
        | 'r' TOK_IMAGE         { $$ = mkun(T_Read,  $2);        }
        | '?'                   { $$ = mkempty(T_Help);          }
        | 'f'                   { $$ = mkempty(T_List);          }
        | 'u'                   { $$ = mkempty(T_Undo);          }
        | 'q'                   { $$ = mkempty(T_Quit);          }

dst     : expr                  { $$ = mkMapcoord(0, NULL, $1);  }

mapdst  : TOK_NEW coord '=' expr { $$ = mkMapcoord(getImageNo($1), $2, $4); }

special : TOK_COLOR             { $$ = mkempty(T_Color);         }
        | TOK_COMB image        { $$ = mkun(T_Comb, $2);         }
        | TOK_DEBUG logfile     { $$ = mkun(T_Debug, $2);        }
        | TOK_DISPLAY display   { $$ = mkun(T_Display, $2);      }
        | TOK_EXPLO image       { $$ = mkun(T_Explo, $2);        }
        | TOK_FREE image        { $$ = mkun(T_Free, $2);         }
        | TOK_GENEPSON dfile image { $$ = mk(T_Genep, $2, $3);   }
        | TOK_GENPS dfile image { $$ = mk(T_Genps, $2, $3);      }
        | TOK_GET rgbcomp image { $$ = mk(T_Get, $2, $3);        }
        | TOK_GRAY              { $$ = mkempty(T_Gray);          }
        | TOK_HELP              { $$ = mkempty(T_Help);          }
        | TOK_LIST              { $$ = mkempty(T_List);          }
        | TOK_LOGFILE logfile   { $$ = mkun(T_Logfile, $2);      }
        | TOK_LUMA image        { $$ = mkun(T_Luma, $2);         }
        | TOK_MATTE image number { $$ = mk(T_Matte, $2, $3);     }
        | TOK_MELT image        { $$ = mkun(T_Melt, $2);         }
        | TOK_MONO              { $$ = mkempty(T_Mono);          }
        | TOK_OFMT dir          { $$ = mkun(T_Ofmt, $2);         }
        | TOK_OIL image         { $$ = mkun(T_Oil, $2);          }
        | TOK_PUT rgbcomp image { $$ = mk(T_Put, $2, $3);        }
        | TOK_QUIT              { $$ = mkempty(T_Quit);          }
        | TOK_RANGE policy      { $$ = mkun(T_Range, $2);        }
        | TOK_READ dfile image  { $$ = mk(T_Read, $2, $3);       }
        | TOK_SHEAR image       { $$ = mkun(T_Shear, $2);        }
        | TOK_SIGNED dir        { $$ = mkun(T_Signed, $2);       }
        | TOK_SIZE number number { $$ = mk(T_Size, $2, $3);      }
        | TOK_SLICE image       { $$ = mkun(T_Slice, $2);        }
        | TOK_STORE image       { $$ = mkun(T_Store, $2);        }
        | TOK_TILE image        { $$ = mkun(T_Tile, $2);         }
        | TOK_TRUNC dir         { $$ = mkun(T_Trunc, $2);        }
        | TOK_UNDO              { $$ = mkempty(T_Undo);          }
        | TOK_VERBOSE dir       { $$ = mkun(T_Verbose, $2);      }
        | TOK_VERSION           { $$ = mkempty(T_Ver);           }
        | TOK_WRITE dfile image { $$ = mk(T_Write, $2, $3);      }

expr    : co
        | expr ';' co           { $$ = mk(T_Comma, $1, $3);      }

co      : as
        | as '?' co ':' co      { $$ = mkCond($1, $3, $5);       }

as      : ss
        | TOK_VAR '=' ss        { $$ = mkAssign($1, $3);         }

ss      : or
        | ss TOK_AND or         { $$ = mk(T_Land, $1, $3);       }
        | ss TOK_OR or          { $$ = mk(T_Lor, $1, $3);        }

or      : xor
        | or '|' xor            { $$ = mk(T_Or, $1, $3);         }

xor     : and
        | xor '^' and           { $$ = mk(T_Xor, $1, $3);        }

and     : eq
        | and '&' eq            { $$ = mk(T_And, $1, $3);        }

eq      : rel
        | eq TOK_EQ rel         { $$ = mk(T_Eq, $1, $3);         }
        | eq TOK_NE rel         { $$ = mk(T_Ne, $1, $3);         }

rel     : shift
        | rel '>' shift         { $$ = mk(T_Gt, $1, $3);         }
        | rel '<' shift         { $$ = mk(T_Lt, $1, $3);         }
        | rel TOK_GE shift      { $$ = mk(T_Ge, $1, $3);         }
        | rel TOK_LE shift      { $$ = mk(T_Le, $1, $3);         }

shift   : plus
        | shift TOK_LSHIFT plus { $$ = mk(T_Lshift, $1, $3);     }
        | shift TOK_RSHIFT plus { $$ = mk(T_Rshift, $1, $3);     }

plus    : times
        | plus '+' times        { $$ = mk(T_Add, $1, $3);        }
        | plus '-' times        { $$ = mk(T_Sub, $1, $3);        }

times   : pow
        | times '*' pow         { $$ = mk(T_Mul, $1, $3);        }
        | times '/' pow         { $$ = mk(T_Div, $1, $3);        }
        | times '%' pow         { $$ = mk(T_Mod, $1, $3);        }

pow     : factor
        | pow TOK_POW factor    { $$ = mk(T_Pow, $1, $3);        }

factor  : fncall2
        | 'x'                   { $$ = mkempty(T_Xcoord);        }
        | 'y'                   { $$ = mkempty(T_Ycoord);        }
        | 'r'                   { $$ = mkempty(T_Prad);          }
        | 'a'                   { $$ = mkempty(T_Pang);          }
        | TOK_ICON              { $$ = mkInum(lastinum);         }
        | TOK_FCON              { $$ = mkFnum(lastfnum);         }
        | '+' factor            { $$ = $2;                       }
        | '-' factor            { $$ = mkun(T_Neg, $2);          }
        | '!' factor            { $$ = mkun(T_Bang, $2);         }
        | '~' factor            { $$ = mkun(T_Not, $2);          }
        | unaryop factor        { $$ = mkun((enum tree) $1, $2); }
        | file coord            { $$ = mkCoord($1, $2);          }
        | TOK_VAR               { $$ = mkVar($1);                }
        | '(' expr ')'          { $$ = $2;                       }

fncall4 : fncall2
        | TOK_DIST  '(' expr ',' expr ',' expr ',' expr ')'
                    { if (!($$ = mkBuiltin($1, 4, $3, $5, $7, $9))) YYERROR; }
        | TOK_ANGLE '(' expr ',' expr ',' expr ',' expr ')'
                    { if (!($$ = mkBuiltin($1, 4, $3, $5, $7, $9))) YYERROR; }

fncall2 : fncall1
        | TOK_HYPOT '(' expr ',' expr ')'
                    { if (!($$ = mkBuiltin($1, 2, $3, $5))) YYERROR; }
        | TOK_ATAN  '(' expr ',' expr ')'
                    { if (!($$ = mkBuiltin($1, 2, $3, $5))) YYERROR; }
 
fncall1 : fncall0
        | TOK_ABS  '(' expr ')' { if (!($$ = mkBuiltin($1, 1, $3))) YYERROR; }
        | TOK_COS  '(' expr ')' { if (!($$ = mkBuiltin($1, 1, $3))) YYERROR; }
        | TOK_LOG  '(' expr ')' { if (!($$ = mkBuiltin($1, 1, $3))) YYERROR; }
        | TOK_SIN  '(' expr ')' { if (!($$ = mkBuiltin($1, 1, $3))) YYERROR; }
        | TOK_SQRT '(' expr ')' { if (!($$ = mkBuiltin($1, 1, $3))) YYERROR; }
 
fncall0 : TOK_RAND '(' ')'      { if (!($$ = mkBuiltin($1, 0, 0)))  YYERROR; }

unaryop : '!'                   { $$ = mkempty(T_Bang);          }
        | '-'                   { $$ = mkempty(T_Neg);           }
        | '~'                   { $$ = mkempty(T_Not);           }

display : /* empty */           { $$ = NULL;                     }
        | dir
        | image

logfile : /* empty */           { $$ = NULL;                     }
        | dir
        | dfile

number  : /* empty */           { $$ = NULL;                     }
        | TOK_ICON              { $$ = mkInum(lastinum);         }
        | TOK_FCON              { $$ = mkFnum(lastfnum);         }

file    : /* empty */           { $$ = 0;                        }
        | TOK_IMAGE             { $$ = getImageNo(lastimage);    }
        | '$' TOK_ICON          { $$ = lastinum + 1;             }

image   : /* empty */           { $$ = NULL;                     }
        | TOK_IMAGE             { $$ = mkInum(getImageNo(lastimage));  }
        | '$' TOK_ICON          { $$ = mkInum(lastinum + 1);     }

dfile   : TOK_FNAME             { $$ = mkempty(T_File);          }

coord   : /* empty */           { $$ = NULL;                     }
        | '{' expr ',' expr '}' { $$ = mk(T_Polar, $2, $4);      }
        | '[' expr ',' expr ']' { $$ = mk(T_Add, $2, mk(T_Mul, $4,
                                          mkInum(CurNew->height))); }

policy  : TOK_RCUT              { $$ = mkempty(T_Rcut);          }
        | TOK_RMINMAX           { $$ = mkempty(T_Rminmax);       }
        | TOK_RWRAP             { $$ = mkempty(T_Rwrap);         }

rgbcomp : TOK_RED               { $$ = mkInum(0);                }
        | TOK_GREEN             { $$ = mkInum(1);                }
        | TOK_BLUE              { $$ = mkInum(2);                }

dir     : /* empty */           { $$ = NULL;                     }
        | '+'                   { $$ = mkempty(T_Plus);          }
        | '-'                   { $$ = mkempty(T_Minus);         }

%%

static struct NameToken SpecialToks[] = {      /* Also update popi.man. */
    { "color",       TOK_COLOR    },
    { "combine",     TOK_COMB     },
    { "debug",       TOK_DEBUG    },
    { "display",     TOK_DISPLAY  },
    { "exit",        TOK_QUIT     },
    { "explode",     TOK_EXPLO    },
    { "free",        TOK_FREE     },
    { "genepson",    TOK_GENEPSON },
    { "genps",       TOK_GENPS    },
    { "get",         TOK_GET      },
    { "grayscale",   TOK_GRAY     },
    { "help",        TOK_HELP     },
    { "list",        TOK_LIST     },
    { "logfile",     TOK_LOGFILE  },
    { "luma",        TOK_LUMA     },
    { "matte",       TOK_MATTE    },
    { "melt",        TOK_MELT     },
    { "monochrome",  TOK_MONO     },
    { "ofmt",        TOK_OFMT     },
    { "oil",         TOK_OIL      },
    { "put",         TOK_PUT      },
    { "quit",        TOK_QUIT     },
    { "range",       TOK_RANGE    },
    { "read",        TOK_READ     },
    { "shear",       TOK_SHEAR    },
    { "signed",      TOK_SIGNED   },
    { "size",        TOK_SIZE     },
    { "slice",       TOK_SLICE    },
    { "store",       TOK_STORE    },
    { "tile",        TOK_TILE     },
    { "truncate",    TOK_TRUNC    },
    { "undo",        TOK_UNDO     },
    { "verbose",     TOK_VERBOSE  },
    { "version",     TOK_VERSION  },
    { "write",       TOK_WRITE    },
    { (char *) 0,    0            }
};

static struct NameToken FuncToks[] = {         /* Also update popi.man. */
    { "abs",         TOK_ABS      },
    { "angle",       TOK_ANGLE    },
    { "atan",        TOK_ATAN     },
    { "cos",         TOK_COS      },
    { "dist",        TOK_DIST     },
    { "hypot",       TOK_HYPOT    },
    { "log",         TOK_LOG      },
    { "rand",        TOK_RAND     },
    { "sin",         TOK_SIN      },
    { "sqrt",        TOK_SQRT     },
    { (char *) 0,    0            }
};

static struct NameToken KeywordToks[] = {      /* Also update popi.man. */
    { "blue",        TOK_BLUE     },
    { "cut",         TOK_RCUT     },
    { "green",       TOK_GREEN    },
    { "minmax",      TOK_RMINMAX  },
    { "new",         TOK_NEW      },
    { "red",         TOK_RED      },
    { "wrap",        TOK_RWRAP    },
    { (char *) 0,    0            }
};


static int
Getch()
{
    int c;

    if (InputStream == (FILE *) 0) {
        if (SaveChar == '\0') {
            c = disp_getchar();
        } else { 
            c = SaveChar;
            SaveChar = '\0';
        }
    } else {
        c = getc(InputStream);
    }
 
    OldPos = CharPos;
           
    if (c == '\t') {
        CharPos = (CharPos - 1) % 8 + 8;
    } else if (c == '\n') {
        CharPos = 0;
    } else {
        ++CharPos;
    }

    DEBUG((Debug, "Getch() => '%c'\n", c));

    if (LogStr && CharPos != OldPos) {
        PUTC(c, LogStr);
    }

    return(c);
}


static int
GetNumber(int first)
{
    double fract;
    int c, num;

    num = first - '0';
    fract = 0.0;
    while (isdigit(c = Getch())) {
        num = 10 * num + c - '0';
    }

    if (c == '.') {
        double div = 10.0;

        while (isdigit(c = Getch())) {
            fract += (c - '0') / div;
            div *= 10.0;
        }
    }
    pushback(c);
    if (fract == 0.0) {
        lastinum = num;
        return(TOK_ICON);
    }
    lastfnum = (double) num + fract;

    return(TOK_FCON);
}


static int
emit(int token)
{
    HaveToken = 1;

    return(token);
}
 
 
static void
pushback(int c)
{
    DEBUG((Debug, "pushback('%c')\n", c));
    if (InputStream == (FILE *) 0) {
        SaveChar = c;
    } else {
        UNGETC(c, InputStream);
    }
    CharPos = OldPos;
}


static int
is_word(enum word_type w, char *buf)
{
    struct NameToken *nt = NULL;
    struct NameToken *start = NULL;
    int cnt, len; 
    int token = 0;

    if (w == W_Special) {
        start = nt = &SpecialToks[0];
    } else if (w == W_Function) {
        start = nt = &FuncToks[0];
    } else if (w == W_Keyword) {
        start = nt = &KeywordToks[0];
    }
    for (cnt = 0; nt->name; nt++) {
        if (strncmp(buf, nt->name, strlen(buf)) == 0) {  
            token = nt->token;
            cnt++;
        }  
    }
    if (!cnt) {
        return(0);
    }
    if (cnt > 1) {
        SPRINTF(ErrBuf, "Operation '%s' ambiguous (", buf);
        for (nt = start; nt->name; nt++) {
            if (strncmp(buf, nt->name, strlen(buf)) == 0) {
                STRCAT(ErrBuf, nt->name);
                STRCAT(ErrBuf, ", ");
            }
        }
        len = strlen(ErrBuf);
        ErrBuf[len-2] = ')';
        ErrBuf[len-1] = '\n';
        error(ERR_PARSE);
        return(0);
    }

    return(token);
}


static int
yylex()
{
    char *l;
    int c, i, special;

    while ((c = Getch()) == ' ' || c == '\t') {
        continue;
    }
    TokPos = CharPos;
    switch (c) {
        case EOF :
            saw_eof = 1;
            RET(0);

        case ':' :
            special = c;
            if (HaveToken) {
                RET(c);
            }
            l = buf;
            while (isalnum(c = Getch())) {
                *l++ = c;
            }
            pushback(c);
            *l = '\0';
            if ((i = is_word(W_Special, buf))) {
                RET(i);
            }
            RET(special);

        case '#' :
            do {
                c = Getch();
            } while (c != '\n');
            pushback('\n');
            RET(TOK_COMMENT);

        case '\n' :
        case '?' :
        case '-' :
        case '+' :
        case '/' :
        case '%' :
        case '~' :
        case '^' :
        case '(' :
        case ')' :
        case '[' :
        case ']' :
        case '{' :
        case '}' :
        case ',' :
        case '$' :
        case ';' :
            RET(c);

        case '*' :
            if ((c = Getch()) == '*') {
                RET(TOK_POW);
            }
            pushback(c);
            RET('*');

        case '&' :
            if ((c = Getch()) == '&') {
                RET(TOK_AND);
            }
            pushback(c);
            RET('&');

        case '|' :
            if ((c = Getch()) == '|') {
                RET(TOK_OR);
            }
            pushback(c);
            RET('|');

        case '>' :
            if ((c = Getch()) == '>') {
                RET(TOK_RSHIFT);
            } else if (c == '=') {
                RET(TOK_GE);
            }
            pushback(c);
            RET('>');

        case '<' :
            if ((c = Getch()) == '<') {
                RET(TOK_LSHIFT);
            } else if (c == '=') {
                RET(TOK_LE);
            }
            pushback(c);
            RET('<');

        case '=' : 
            if ((c = Getch()) == '=') {
                RET(TOK_EQ);
            }
            pushback(c);
            RET('=');

        case '!' : 
            if ((c = Getch()) == '=') {
                RET(TOK_NE);
            }
            pushback(c);
            RET('!');

        case '\"' : 
            l = lastfname;
            while ((c = Getch()) != '\"' && c != '\n' && c != EOF) {
                *l++ = c;
            }
            if (c != '\"') {
                pushback(c);
            }
            *l = '\0';
            yylval.name = STRDUP(lastfname);
            RET(TOK_FNAME);

        default : 
            RET(parse_default(c));
    }
}


static int
parse_default(int c)
{
    char *l;

    if (isdigit(c)) {
        return(GetNumber(c));
    } else {
        l = lastimage;
        for (;;) {
            *l++ = c;
            switch (c = Getch()) {
                case EOF  :
                case ' '  :
                case '\t' :
                case '\n' :
                case '?'  :
                case ':'  :
                case '-'  :
                case '+'  :
                case '*'  :
                case '/'  :
                case '%'  :
                case '~'  :
                case '^'  :
                case '('  :
                case ')'  :
                case '<'  :
                case '>'  :
                case '['  :
                case ']'  :
                case '&'  :
                case '|'  :
                case '='  :
                case '!'  :
                case '\"' :
                case ','  :
                case ';'  :
                case '#'  :
                case '{'  :
                case '}'  : 
                    pushback(c);
                    *l = '\0';
                    return(check_buf());
            }
        }
    }
}


static int
check_buf()
{
    int i;
    int reply = TOK_IMAGE;

    if (strlen(lastimage) == 1) {
        switch (*lastimage) {  
            case 'x' :
            case 'y' :
            case 'u' :
            case 'f' :
            case 'q' :
            case 'w' :
            case 'r' :
            case 'a' : 
                return(*lastimage);

            case 'X' : 
                lastinum = CurNew->width - 1;
                return(TOK_ICON);

            case 'Y' : 
                lastinum = CurNew->height - 1;
                return(TOK_ICON);

            case 'A' : 
                lastinum = MAX_ANG;
                return(TOK_ICON);

            case 'R' : 
                lastinum = dist(0, 0, CurNew->width / 2, CurNew->height / 2);
                return(TOK_ICON);

            case 'Z' : 
                lastinum = Zmax;
                return(TOK_ICON);

            case 'T' : 
                lastinum = MAX_TRIG;
                return(TOK_ICON);
        }
    }

    if ((i = is_word(W_Keyword, lastimage))) {
        reply = i;
    } else if ((i = is_word(W_Function, lastimage))) {
        reply = i;
    }
    if (reply != TOK_IMAGE) {
        yylval.name = STRDUP(lastimage);
    }

    return(reply);
}


static void
yyerror(char *err)
{
    STRCPY(ErrBuf, err);
    error(ERR_PARSE);
}
